/* Original code by Bushrod Thomas.
 * Ported to BeOS by CGP.
 * CGP, 11/13/00
 *	Inputs and outputs from this network appear to require
 *   unit values between -1.0 and +1.0
 */

#include "ANN.h"
#include "COb.XPlex.h"
#include <time.h>		// for RAND (used to initialize the weights)
#include <stdlib.h>		// for RAND
#include <iostream.h>           // for cout/cin
#include <math.h>		// for exp 
#include <Debug.h> 		// from BeOS for ASSERT

/////////////////////////////////////
//       NeuronActivity class      //
/////////////////////////////////////

// constructor
NeuronActivity::NeuronActivity(ACTIVITY_TYPE a)
: activity( a )
{
}

// destructor
NeuronActivity::~NeuronActivity()
{
}

/////////////////////////////////////
//        Neuron class             //
/////////////////////////////////////

// constructor
Neuron::Neuron()
: activity( 0.0 )
{
}

// destructor
Neuron::~Neuron()
{
  for( int i = 0; i < dendrite.length(); i++ )
  {
    delete dendrite[i];  // get rid of all the Synapse objects
  }
  dendrite.clear(); // get rid of pointers
}


/////////////////////////////////////
//        Synapse class            //
/////////////////////////////////////

// constructor
Synapse::Synapse(Neuron* pFN)
: p_from_neuron( pFN )
{   
  // set the weight to a random number between -1 and 1. (10000 gives 5 digit precision)
  weight = ((WEIGHT_TYPE)(rand()%10000) / (WEIGHT_TYPE)10000 );
  if (rand() % 2 == 1) weight = - weight;
  
  summedWeightChange = 0.0;
}

// destructor
Synapse::~Synapse()
{
}

// this function computes the weighted ouput value produced from this synapse.
ACTIVITY_TYPE Synapse::Influence()
{
  //cout << "weight " << weight << " * p_from_neuron->activity " << p_from_neuron->activity;//*****
  //cout << " = " << weight * p_from_neuron->activity << "\n";//*****
  
  return weight * p_from_neuron->activity;
}


/////////////////////////////////////
//          Layer class            //
/////////////////////////////////////

// constructor
Layer::Layer()
{
}

// destructor
Layer::~Layer()
{
  for( int i = 0; i < neurons.length(); i++ )
  {
    Neuron* aNeuron = (Neuron*)neurons[i];
    delete aNeuron; // kill all of the neuron objects.
  }
  neurons.clear(); // get rid of pointers
}


/////////////////////////////////////
//            ANN class            //
/////////////////////////////////////

// constructor
ANN::ANN(int n_neurons_in_input_layer, int n_neurons_in_output_layer)
{
  n_samples_analyzed = 0;
  
  // initialize the random number generator for use initializing weights.
  srand( (unsigned)time( NULL ) );
  
  // create the input layer
  Layer* inputLayer = new Layer();
  int i;
  for (i=0; i < n_neurons_in_input_layer; i++)
  {
    inputLayer->neurons.add_high( new Neuron() );
  }
  // put the inputLayer in the list of layers.
  layers.prepend( inputLayer ); // add to head or front of the list
  
  // create the output layer
  Layer* outputLayer = new Layer();
  for ( i=0; i < n_neurons_in_output_layer; i++)
  {
    Neuron* theNeuron = new Neuron();
    for (int j=0; j < n_neurons_in_input_layer; j++)
    {
      // connect this Neuron to a Synapse that connects to a node in the inputLayer.
      theNeuron->dendrite.add_high( new Synapse( (Neuron*)inputLayer->neurons[j] ) );
    }			
    outputLayer->neurons.add_high( theNeuron );
  }
  // put the outputLayer in the list of layers.
  layers.append( outputLayer );
}

// destructor
ANN::~ANN()
{
  for( int i = 0; i < summedWeightChanges.length(); i++ )
  {
    delete summedWeightChanges[i];
  }
  summedWeightChanges.clear(); // get rid of pointers
  
  while (layers.length() > 0)
  {
    Layer* aLayer = (Layer*)layers.remove_front();
    delete aLayer;
  }
  layers.clear(); // I don't think this is necessary.
}

inline ACTIVITY_TYPE ANN::SigmoidActivationFunction( ACTIVITY_TYPE x )
{
  // the logistic function
  return 1.0 / (1.0 + exp( (double) - x ) );
}

// add a hidden layer to the network.  (This is a way to specify how many neurons there are in each
// hidden layer, without having to have a variable number of parameters in the ANN constructor).
// The new hidden layer is inserted right before the output layer.
void ANN::AddHiddenLayer(int n_neurons_in_hidden_layer)
{   
  // it doesn't make sense to add a hidden layer with no neurons!
  if (n_neurons_in_hidden_layer > 0)
  {
    Pix pos = layers.last(); // points to the outputLayer
    Layer*	outputLayer = (Layer*)layers(pos);            // the last layer

    layers.prev(pos);  // set the pointer to the previous layer
    Layer*	oldPenultimateLayer = (Layer*)layers(pos);    // the next-to-last layer

    Layer*	newHiddenLayer = new Layer();                 // the NEW next-to-last layer
    

    int n_neurons_in_outputLayer = outputLayer->neurons.length(),                 
      n_neurons_in_old_penultimate_layer = oldPenultimateLayer->neurons.length();
    
    // create the new layer
    int i, j;
    for (i=0; i < n_neurons_in_hidden_layer; i++)
    {
      Neuron* newNeuron = new Neuron();
      
      // link the new neuron to each of the neurons in the previous layer (via a Synapse)
      for (j=0; j < n_neurons_in_old_penultimate_layer; j++)
      {
	newNeuron->dendrite.add_high( new Synapse( (Neuron*)oldPenultimateLayer->neurons[j] ) );
      }
      newHiddenLayer->neurons.add_high( newNeuron );
    }
    // insert the new Hidden Layer into the list of layers (right before the outputLayer).
    //    (pos still points to the next to last layer)
    layers.ins_after( pos, newHiddenLayer );
    
    // disconnect the outputLayer from the oldPenultimateLayer, and connect it to the newHiddenLayer
    for (i=0; i < n_neurons_in_outputLayer; i++)
    {
      Neuron* theOutputNeuron = (Neuron*)outputLayer->neurons[i];
      
      // get rid of this outputNeuron's synapses that connect to the oldPenultimateLayer
      for (j=0; j < n_neurons_in_old_penultimate_layer; j++)
      {
	delete theOutputNeuron->dendrite[j];
      }
      theOutputNeuron->dendrite.clear();
      
      // connect this outputNeuron to the newHiddenLayer.
      for (j=0; j < n_neurons_in_hidden_layer; j++)
      {
	theOutputNeuron->dendrite.add_high( new Synapse( (Neuron*)newHiddenLayer->neurons[j] ) );
      }
    }
  }
}

// Use this to compute the weight changes for the sample input (and add them to the 
// running total for the previously "analyzed" training samples).
void ANN::Analyze(CObXPlex* sample, CObXPlex* desiredOutputs, double rate)
{
  // compute the resulting (observed) output by evaluating the sample
  CObXPlex* observedOutputs = Evaluate( sample ); 
  
  // compute the benefit of changing the nodes in the output (rightmost) layer (desired - observed).
  CObXPlex* rightBenefits = new CObXPlex;
  int n_of_neurons_in_output_layer = observedOutputs->length();
  
  //cout << "\n";//*****
  for (int i=0; i < n_of_neurons_in_output_layer; i++)
  {
    NeuronActivity* theDesiredActivity  = (NeuronActivity*)((*desiredOutputs)[i]);
    NeuronActivity* theObservedActivity = (NeuronActivity*)((*observedOutputs)[i]);
    
    ACTIVITY_TYPE benefit = theDesiredActivity->activity - theObservedActivity->activity;
    
    //cout << "OutputNeuron " << i << " Desired " << theDesiredActivity->activity;//*****
    //cout << " Observed " << theObservedActivity->activity << " Benefit (De-Ob) ";//*****
    //cout << benefit << "\n";//*****
    
    delete theObservedActivity;
    
    rightBenefits->add_high( new NeuronActivity( benefit ) );
  }
  observedOutputs->clear();
  delete observedOutputs;
  
  //cout << "\n";//*****
  //int layerIndex = layers.GetCount()-2; //*****
  
  // Compute the benefits for all other layers, starting with the layer just left of output,
  //   and compute the weight changes for all synapse weights (layer by layer).
  CObXPlex* leftBenefits = NULL;
  Pix pos = layers.last();
  Layer* rightLayer = (Layer*)layers(pos); // picturing the net feeding from left to right,
  // each layer's benefits help determine the
  // benefits for the layer just to its left.

  layers.prev(pos);    // set the pointer to the previous layer
  while( pos != 0 ) 
  {
    //cout << "Synapses after Layer " << layerIndex-- << "\n";//*****
    
    Layer* leftLayer = (Layer*)layers(pos); // starting with the penultimate layer.
    layers.prev(pos);    // set the pos pointer to the previous layer

    int n_neurons_in_left_layer  =  leftLayer->neurons.length();
    int n_neurons_in_right_layer = rightLayer->neurons.length();
    
    // with the exception of the input layer (no need to compute benefits),
    // for each neuron in the left layer, compute benefits...
    if ( leftLayer != layers.front() ) // head
    {
      leftBenefits = new CObXPlex;
      
      for (int j=0; j < n_neurons_in_left_layer; j++)
      {	
	// unused variable, CGP, 6/15/00
	// Neuron* leftNeuron = (Neuron*)leftLayer->neurons[j];
	ACTIVITY_TYPE leftBenefit = 0.0 ;
	
	// calculate the benefit of changing each neuron in the left layer
	// based on the benefit of changing each (connected) neuron in the right layer.
	for (int k=0; k < n_neurons_in_right_layer; k++)
	{
	  Neuron* rightNeuron = (Neuron*)rightLayer->neurons[k];	
	  Synapse* theSynapse = (Synapse*)rightNeuron->dendrite[j]; // between right & left
	  
	  ACTIVITY_TYPE rightObserved = rightNeuron->activity;
	  ACTIVITY_TYPE rightBenefit  = ((NeuronActivity*)((*rightBenefits)[k]))->activity; 
	  
	  //cout << " Wj->k " << j << "->" << k << " weight " << theSynapse->weight;//*****
	  //cout << "\n	Ok " << rightObserved << " 1-Ok " << 1-rightObserved;//*****
	  //cout << " Bk " << rightBenefit << " w*Ok*(1-Ok)*Bk " << //*****
	  // (theSynapse->weight * rightObserved * ( 1 - rightObserved ) * rightBenefit);//*****
	  //cout << "\n";//*****
	  
	  leftBenefit += (theSynapse->weight * rightObserved * 
			  ( 1 - rightObserved ) * rightBenefit);
	}
	leftBenefits->add_high( new NeuronActivity( leftBenefit ) );
	//cout << "Total_Bj " << leftBenefit << "\n\n";//*****	
      }
    }			
    
    // compute the weight changes for synapses between the left & right layers			
    for ( int j=0; j < n_neurons_in_right_layer; j++)
    {
      NeuronActivity* rightNeuronActivity = (NeuronActivity*)((*rightBenefits)[j]);
      ACTIVITY_TYPE rightBenefit = rightNeuronActivity->activity;
      delete rightNeuronActivity; // this is the last time we will need to access this rightBenefit.
      
      Neuron* rightNeuron =  (Neuron*)rightLayer->neurons[j];
      ACTIVITY_TYPE rightObserved = rightNeuron->activity;
      
      for (int i=0; i < n_neurons_in_left_layer; i++)
      {
	Neuron*  leftNeuron =  (Neuron*)leftLayer->neurons[i];
	ACTIVITY_TYPE leftObserved = leftNeuron->activity;
	
	//cout << " WGT-CHNG: rate " << rate << " Oi " << leftObserved;//*****
	//cout << " Oj " << rightObserved << " (1-Oj) " << (1-rightObserved);//*****
	//cout << " Bj " << rightBenefit << "\n	Delta Wi->j (";//*****
	
	WEIGHT_TYPE change = rate * leftObserved * rightObserved *
	  ( 1 - rightObserved ) * rightBenefit;
	
	//cout << i << "->" << j << ") = r*Oi*Oj*(1-Oj)*Bj = " << change << "\n";//*****
	
	// sum the benefits
	Synapse* theSynapse = (Synapse*)rightNeuron->dendrite[i];
	theSynapse->summedWeightChange += change;
	//cout << "	summedWeightChange = " << theSynapse->summedWeightChange << "\n\n"; //*****
      }
    }
    rightBenefits->clear();
    delete rightBenefits;  // get rid of the array of right layer benefits
    
    ASSERT(leftBenefits != NULL);
    rightBenefits = leftBenefits;  // move the focus one layer left
    rightLayer = leftLayer;
  }
  n_samples_analyzed++;
}

// Print the weights to the outStream
void ANN::OutputWeights(fstream& outStream)
{  
  int layerCount = layers.length();
  
  outStream << "Weights of network\n";
  // outStream << "layerCount = " << layerCount << "\n";
  
  Pix pos = layers.last();
  Layer* rightLayer = (Layer*)layers(pos); // picturing the net feeding from left to right,

  layers.prev(pos);    // set the pointer to the previous layer
  while( pos != 0 ) 
  {
    Layer* leftLayer = (Layer*)layers(pos); // starting with the penultimate layer.
    layers.prev(pos);    // set the pos pointer to the previous layer

    int n_neurons_in_left_layer  =  leftLayer->neurons.length();
    int n_neurons_in_right_layer = rightLayer->neurons.length();
    
    outStream << "Layer: " << layerCount << "\n";
    layerCount--;
    
    for (int j=0; j < n_neurons_in_left_layer; j++)
    {		
	for (int k=0; k < n_neurons_in_right_layer; k++)
	{
	  Neuron* rightNeuron = (Neuron*)rightLayer->neurons[k];	
	  Synapse* theSynapse = (Synapse*)rightNeuron->dendrite[j]; // between right & left
	  
	  outStream << "w[" << j << "," << k << "]= " << theSynapse->weight << "\n";
	}
    }
    
    rightLayer = leftLayer;
  }
}

// this member function is used to actually change the weights for the network.
// Use this after all the training examples have been 'Analyze'd.
void ANN::Train()
{
  // make sure there is a weight change available (should be at least one Analyze call
  // before calling Train, otherwise the change is zero).
  if (n_samples_analyzed > 0 )
  {
    Pix pos = layers.first();
    layers.next(pos);        // skip the input layer (no synapses influence it).

    Layer* aLayer;
    int layerIndex = 0;//*****	                               
    
    // this function assumes that the Analyze function has created the set of weight changes
    // and stored them in summedWeightChange of each Synapse object where the weight is to be changed.
    while( pos != NULL )
    {
      aLayer = (Layer*)layers(pos);
      layers.next(pos);
      
      int n_neurons_in_this_layer = aLayer->neurons.length();
      
      for (int i=0; i < n_neurons_in_this_layer; i++)
      {	
	Neuron* aNeuron = (Neuron*)aLayer->neurons[i];
	
	int n_synapses_influencing_this_neuron = aNeuron->dendrite.length();
	
	for (int j=0; j < n_synapses_influencing_this_neuron; j++)
	{
	  Synapse* aSynapse = (Synapse*)aNeuron->dendrite[j];
	  
	  //cout << "Layer " << layerIndex << " Neuron " << i << " InputSynapse " << j;//*****
	  //cout << "	oldWeight " << aSynapse->weight << "\n";//*****
	  
	  aSynapse->weight += (aSynapse->summedWeightChange / n_samples_analyzed);
	  
	  //cout << " summedWeightChange " << aSynapse->summedWeightChange;//*****
	  //cout << " n_samples_analyzed " << n_samples_analyzed;//*****
	  //cout << " sumWC/n_s_a " << aSynapse->summedWeightChange / n_samples_analyzed;//*****
	  //cout << " newWeight " << aSynapse->weight << "\n\n";//*****
	  
	  aSynapse->summedWeightChange = 0.0;
	}		 			
      }
      layerIndex++;//*****
    }
    n_samples_analyzed = 0;
  }
}

// use evaluate to find the result of running the given sample through the net.
CObXPlex* ANN::Evaluate(CObXPlex* sample)
{
  CObXPlex* theResult = new CObXPlex();
  
  Layer* theInputLayer = (Layer*)layers.front(); // head
  int n_of_neurons_in_input_layer = theInputLayer->neurons.length();
  
  //cout << "\n"; //*****
  
  // put the sample into the input layer neurons
  int i;
  for (i=0; i < n_of_neurons_in_input_layer; i++)
  { 
    Neuron* anInputNeuron = (Neuron*)theInputLayer->neurons[i];
    NeuronActivity* theSampleActivity = (NeuronActivity*)((*sample)[i]);
    
    anInputNeuron->activity = theSampleActivity->activity;
    
    //cout << "inputLayer Neuron " << i << " = " << anInputNeuron->activity << "\n"; //*****
  }
  
  int layerNumber = 1; //*****
  
  // propagate the activities through the network
  Pix pos = layers.first();
  layers.next(pos);  // skip this first layer

  Layer* theCurrentLayer;
  // go through each layer in the net
  while (pos != 0)
  {
    theCurrentLayer = (Layer*)layers(pos);
    layers.next(pos);

    // go through all the neurons in this layer
    int n_of_neurons_in_this_layer	= theCurrentLayer->neurons.length();
    for (i=0; i < n_of_neurons_in_this_layer; i++)
    {
      Neuron* theCurrentNeuron = (Neuron*)theCurrentLayer->neurons[i];
      int n_synapses_influencing_the_current_neuron = theCurrentNeuron->dendrite.length();
      
      //cout << "\n Layer # " << layerNumber << "\n";//*****
      
      ACTIVITY_TYPE act = 0.0;				                                                                 
      // go through each incoming synapse for this neuron.
      for ( int j=0; j < n_synapses_influencing_the_current_neuron; j++ )
      {
	Synapse* theCurrentSynapse = (Synapse*)theCurrentNeuron->dendrite[j];
	
	//cout << "Neuron " << i << " synapse " << j << " ";//*****
	act += theCurrentSynapse->Influence();
      }
      // set the activity to the sum of the weighted inputs.
      theCurrentNeuron->activity = SigmoidActivationFunction( act );
      
      //cout << "Neuron " << i << " summed weighted input = " << act;//*****
      //cout << "\n	SigmoidActivityFunction(summed weighted input) = "; //*****
      //cout << theCurrentNeuron->activity << "\n";//*****
    }   
    layerNumber++; //*****
  }        
  
  Layer* theOutputLayer = (Layer*)layers.rear(); // tail
  int n_of_neurons_in_output_layer = theOutputLayer->neurons.length();
  
  // copy the output layer activity to the result array that is to be returned.
  for ( i=0; i < n_of_neurons_in_output_layer; i++)
  {
    Neuron* anOutputNeuron = (Neuron*)theOutputLayer->neurons[i];
    
    theResult->add_high( new NeuronActivity( anOutputNeuron->activity ) );
    //cout << "** outputNeuron " << i << " = " << anOutputNeuron->activity << "\n";
  }
  
  return theResult;
}
